/**
 *
 * \file        Uart_SC16IS740.h
 * \brief       External Uart SC16IS740/750/760 Hardware class Definitions
 * \author      Andrew Salmon
 * \date        07/13/2009
 *
 */
#ifndef _UART_SC16IS740_H_
#define _UART_SC16IS740_H_
#include "dm_nutypedefs.h"

//Uart Register Description
#define SPI_UART_READ_ADDR_MASK    0x80    // SPI uart read address mask
#define SPI_UART_WRITE_ADDR_MASK   0x00    // SPI uart write address mask
#define UART_REG_ADDR_BITNUM       0x03    //adress starts at bit 3

#define UART_THR_ADDR          0x00    //transmit holding register address                 
#define UART_RHR_ADDR          0x00    //receive holding register address
#define UART_IER_ADDR          0x01    //interrupt enable register address
#define UART_IIR_ADDR          0x02    //interrupt identification register address
#define UART_FCR_ADDR          0x02    //FIFO control register address
#define UART_LCR_ADDR          0x03    //Line control register address
#define UART_MCR_ADDR          0x04    //Modem control register address
#define UART_LSR_ADDR          0x05    //Line status register address
#define UART_MSR_ADDR          0x06    //Modem status register address
#define UART_SPR_ADDR          0x07    //Scratch pad register address
#define UART_TCR_ADDR          0x06    //transmission control register address
#define UART_TLR_ADDR          0x07    //trigger level register address
#define UART_TXLVL_ADDR        0x08    //transmit FIFO level register address
#define UART_RXLVL_ADDR        0x09    //receive FIFO level register address
#define UART_IODir_ADDR        0x0A    //I/O pin direction register address
#define UART_IOState_ADDR      0x0B    //I/O pin state register address
#define UART_IOIEN_ADDR        0x0C    //I/O pin interrupt enable register address
#define UART_IOCTRL_ADDR       0x0E    //I/O pin control register address
#define UART_EFCR_ADDR         0x0F    //extra feature register address
#define UART_MAX_ADDR          0x10    //register address limit
#define IsValidRegister(a)          ( ((UINT8)a) < UART_MAX_ADDR )
//special registers
#define UART_DLL_ADDR          0x00    //divisor latch LSB register address
#define UART_DLH_ADDR          0x01    //divisor latch MSB register address
#define UART_EFR_ADDR          0x02    //enhanced feature register address
#define UART_RES_ADDR          0x03    //reserve special register address
#define UART_XON1_ADDR         0x04    //Xon1 word register address
#define UART_XON2_ADDR         0x05    //Xon2 word register address
#define UART_XOFF1_ADDR        0x06    //Xoff1 word register address
#define UART_XOFF2_ADDR        0x07    //Xoff1 word register address

// UIIR Interrupt Identification Register
#define UIIR_IP                 0x01    //  Bit 0: IP (Interrupt pending)
#define UIIR_THRE               0x02    //  Transmit buffer empty interrupt (THRE) (Third priority)
#define UIIR_RDA                0x04    //  Receive data available or Trigger level interrupt (RDA) (Second priority)
#define UIIR_RLS                0x06    //  Receive line status interrupt (Highest priority)
#define UIIR_CTI                0x0C    //  Character timeout interrupt (CTI) (Second priority)
#define UIIR_SC                 0x10    //  receive Xoff signal (special character)
#define UIIR_CTSRTS             0x20    //  CTS, RTS change state

#define UIIR_MSI                0x00    //  Modem status change interrupt (MSI) (lowest priority)
#define UIIR_LC                 0x30    //  Line change
#define UIIR_IID                0x0E    //  Bit 3-1: IID (Interrupt identification)

// ULSR Line Status Register
#define ULSR_RXFE               0x80    //  Receive error
#define ULSR_TEMT               0x40    //  Transmit empty
#define ULSR_THRE               0x20    //  Transmit holding register empty
#define ULSR_BI                 0x10    //  Break indicator
#define ULSR_FE                 0x08    //  Framing error
#define ULSR_PE                 0x04    //  Parity error
#define ULSR_OE                 0x02    //  Overrun error
#define ULSR_DR                 0x01    //  Receiver data ready

// UMCR Modem Control Register
#define UMCR_CD                 0x80    //  Clock Divisor
#define UMCR_IrDA               0x40    //  IrDA mode
#define UMCR_XONA               0x20    //  Xon Any
#define UMCR_LB                 0x10    //  Loopback mode
#define UMCR_RES                0x08    //  Reserved
#define UMCR_TCR                0x04    //  TCR and TLR enable
#define UMCR_RTS                0x02    //  Force RTS enable (active low)
#define UMCR_DTR                 0x01    //  Receiver data ready

// UMCR Modem Status Register
#define UMSR_CD                 0x80    //  
#define UMSR_RI                 0x40    //  
#define UMSR_DSR                0x20    //  
#define UMSR_CTS                0x10    //  CTS (active high)
#define UMSR_CDD                0x08    //  
#define UMSR_RID                0x04    //  
#define UMSR_DSRD               0x02    //  
#define UMSR_CTSD               0x01    //  CTS input changed

#define ULSR_DATA_ERROR         ( ULSR_PE | ULSR_FE | ULSR_BI | ULSR_OE )
#define USART_OVERRUN_ERROR     ULSR_OE
#define USART_PARITY_ERROR      ULSR_PE
#define USART_FRAMING_ERROR     ULSR_FE
#define USART_BREAK_DETECT      ULSR_BI

// UIER Interrupt Enable Register
#define UIER_RDAIE              0x01    //  UART receiver interrupt
#define UIER_THREIE             0x02    //  UART transmitter interrupt
#define UIER_RXLSIE             0x04    //  UART line status interrupt
#define UIER_MSIE               0x08    //  UART modem status interrupt
#define UIER_SM                 0x10    //  UART sleep mode
#define UIER_XOFFIE             0x20    //  UART XOFF interrupt
#define UIER_RTSIE              0x40    //  UART RTS interrupt
#define UIER_CTSIE              0x80    //  UART CTS interrupt

// ULCR Line Control Register
#define ULCR_WLS0               0x01    //word lenght bit 0
#define ULCR_WLS1               0x02    //word length bit 1
#define ULCR_SBS                0x04    //stop bits
#define ULCR_PE                 0x08    //parity enable
#define ULCR_PS0                0x10    //parity type select (odd/even)
#define ULCR_PS1                0x20    //select forced parity
#define ULCR_BC                 0x40    //break control
#define ULCR_DLAB               0x80    //divisor latch enable
  
// UFCR FIFO control register
#define UFCR_FCRFE              0x01    // Enable the whole thing
#define UFCR_RFR                0x02    // Rx FIFO reset
#define UFCR_TFR                0x04    // Tx FIFO reset
#define UFCR_RES                0x08    // reserved
#define UFCR_TTLS0              0x10    // transmit trigger level bits
#define UFCR_TTLS1              0x20
#define UFCR_RTLS0              0x40    // Receive trigger level bits
#define UFCR_RTLS1              0x80

// TCR tx control register
#define UTCR_RFHTLS0              0x01 // hault bit0  
#define UTCR_RFHTLS1              0x02    
#define UTCR_RFHTLS2              0x04    
#define UTCR_RFHTLS3              0x08 // hault bit3 
#define UTCR_RFRTLS0              0x10 // resume bit0   
#define UTCR_RFRTLS1              0x20
#define UTCR_RFRTLS2              0x40    
#define UTCR_RFRTLS3              0x80 // resume bit3  

// UEFR Enhance Features Register
#define UEFR_CTS                0x80    //  CTS enable
#define UEFR_RTS                0x40    //  RTS enable
#define UEFR_Xoff               0x20    //  XOFF enable
#define UEFR_EN                 0x10    //  enhance functions enable
#define UEFR_SFC                0x0f    //  software flow control bits 0-3

// UEFCR Extra Features Register
#define UEFCR_NBIT              0x01
#define UEFCR_RXE               0x02
#define UEFCR_TXE               0x04
#define UEFCR_RES1              0x08
#define UEFCR_ARTSD             0x10
#define UEFCR_ARTSO             0x20
#define UEFCR_RES2              0x40
#define UEFCR_IrDA              0x80   

//Rx FIFO levels
#define RX_FIFO_LEVEL_0         1
#define RX_FIFO_LEVEL_8         8 
#define RX_FIFO_LEVEL_16        16
#define RX_FIFO_LEVEL_56        56
#define RX_FIFO_LEVEL_60        60
#define RX_HALT_RESUME_LEVEL_MAX RX_FIFO_LEVEL_60
/*
FCR[7] (MSB),
FCR[6] (LSB)
     RX trigger. Sets the trigger level for the RX FIFO.
     00 = 8 characters
     01 = 16 characters
     10 = 56 characters
     11 = 60 characters*/

//Tx FIFO levels
#define TX_FIFO_LEVEL_0         0//use default max 
#define TX_FIFO_LEVEL_8         8 
#define TX_FIFO_LEVEL_16        16
#define TX_FIFO_LEVEL_32        32
#define TX_FIFO_LEVEL_56        56
/*
FCR[5] (MSB),
FCR[4] (LSB)
     TX trigger. Sets the trigger level for the TX FIFO.
     00 = 8 spaces
     01 = 16 spaces
     10 = 32 spaces
     11 = 56 spaces
FCR[5:4] can only be modified and enabled when EFR[4] is set. This is
because the transmit trigger level is regarded as an enhanced function.*/

#define SC1_BUS_SPI 0
#define SCI_BUS_I2C 1

/* UART Parity */
#define SCI_UART_Parity_No	      0x0000  /* Parity Disable */
#define SCI_UART_Parity_Even	      0x0006  /* Even Parity */
#define SCI_UART_Parity_Odd	      0x0002  /* Odd Parity */
#define SCI_UART_Parity_OddStick      0x0082  /* 1 is transmitted as bit parity */
#define SCI_UART_Parity_EvenStick     0x0086  /* 0 is transmitted as bit parity */

#define SCI_UART_COMSPEC_CTS	0x0200
#define SCI_UART_COMSPEC_RTS	0x0400
#define SCI_UART_COMSPEC_XOFFT   0x0800
#define SCI_UART_COMSPEC_XOFFR   0x1000
#define SCI_UART_COMSPEC_RCTS 	0x4000   // Report CTS

//Configuration table
struct SC16IS740Cfg
{
    UINT8 bUart;          //uart inst
    UINT8 bBusSelect;     //SPI of I2C bus
    UINT8 bBus;           //which I2C/SPI bus the device is on
    UINT8 bBaseAddr;      //the I2C/SPI address of the chip 
    UINT8 bSerialPortInst; //serialport instance
    UINT8 bEnableFIFOs;   //enable tx/rx fifo
    UINT8 bRxTriggerLevel;//rx fifo trigger level
    UINT8 bTxTriggerLevel;//tx fifo trigger level
    UINT8 bAutoCTS;       //assert interrupt on CTS signal change
    UINT8 bAutoRTS;       //assert interrupt on CTS signal change   
    UINT32   taskPrio;/**< external uart tasks priority */
    UINT32   QueueSize;/**< external uart task queue size */
    BOOL     OwnTask; /**< external uart has own task */
    UINT8 bRxFifoHalt;   // if auto cts/rts halt rx at this fifo level
    UINT8 bRxFifoResume; // if auto cts/rts resume rx at this fifo level
};
class SC16IS740Hw
{
public:
    virtual UINT8 ReadRegister( UINT8 reg ) = 0;
    virtual void  WriteRegister( UINT8 reg, UINT8 value ) = 0;
    void  SetRegisterBits( UINT8 bReg, UINT8 bDataBits );
    void  ClearRegisterBits( UINT8 bReg, UINT8 bDataBits);
    UINT8 GetFCR_Register( void );    

protected:
    UINT8  m_Bus;       //which I2C/SPI bus the device is on
    UINT8  m_BaseAddr;  //the I2C/SPI address of the chip    
    UINT8  m_FCR;//FCR register content..write only..a read is IIR status
    UINT8  m_LCR;//LCR content controls FCR or EFR write
    UINT8  m_EFR;//EFR content 
};
//SPI support
class SC16IS740Hw_SPI : public SC16IS740Hw
{
public:
    SC16IS740Hw_SPI( const SC16IS740Cfg *pHwCfg );
    ~SC16IS740Hw_SPI(){};

    UINT8 ReadRegister( UINT8 reg );
    void  WriteRegister( UINT8 reg, UINT8 value );        

private:

    
};
//I2C support
class SC16IS740Hw_I2C : public SC16IS740Hw
{
public:
    SC16IS740Hw_I2C( const SC16IS740Cfg *pHwCfg );
    ~SC16IS740Hw_I2C(){};

    UINT8 ReadRegister( UINT8 reg );
    void  WriteRegister( UINT8 reg, UINT8 value );           

private:

   
};
class Uart_SC16IS740 
{
public:
    Uart_SC16IS740(const SC16IS740Cfg *pHwCfg );
    ~Uart_SC16IS740(){};
    
    void PrintDebug(void);
    void EnableEdgeInt ( BOOL Edge);
    void InitializeEdgeInt ( BOOL Edge );
    void DisableEdgeInt (void);
    void HwReset (void);
    UINT8 GetCTS(void);
    void AssertRTS( UINT8 state);
    void EnableLoopback( UINT8 state );
    void EnableAutoCTS( UINT8 state );
    void EnableAutoRTS( UINT8 state );
    void SetBaudDivisorLatchWord( UINT8 lsb, UINT8 msb);
    void UART_SendData( UINT8 Data);
    void HwConfig(UINT32 inst, UINT32 bps, UINT32 stopBits, UINT32 parity, UINT32 wlenght, UINT32 hwcontrol);
    void SetBaudrate( UINT32 bps, UINT8 *pLSB, UINT8 *pMSB, UINT8 *pPreScaler);
    void TestTxRx(void);
    void ConsoleTx(UINT8 *data, UINT32 count, UINT32 inst);
    void HwUartTx(UINT32 uartNum, UINT8 data);
    void SetRxFIFOTriggerLevel(void);
    void SetTxFIFOTriggerLevel(void);
    void EnableEnhancedFunctions( UINT8 state);
    void WriteEnahancedFeatureRegister( UINT8 state, UINT8 bDataBits);
    void ProcessIRQEvent( void );
    void PostEvent( void );
    void ProcessUartTask( void );
    BOOL InterruptActiveStatus( void );   
    void ServiceDevice(void);
    void CreateMsgQueue(void);
    void DestroyMsgQueue(void);
    void DisableHardware( void );
    void SetRxFIFOHaultResumeTriggerLevel(void);
    
    const SC16IS740Cfg *m_HwCfg;    

private:
    SC16IS740Hw  *m_pHwAccess;               // pointer to hardware access class  
    BOOL     m_HandshakeCTS;/**< CTS handshake flag received in com spec */
    BOOL     m_HandshakeRTS;/**< RTS handshake flag received in com spec */
    UINT32 m_EventGroup;/**< os index into event queue */  
    UINT32 mQueueCount;  
    UINT32 m_FramingErrorCnt;
    UINT32 m_OverrunErrorCnt;
    UINT32 m_ParityErrorCnt;
    UINT32 m_FIFOErrorCnt;
};
extern void UartSC16IS740ProcessIRQEvent();
extern void UartSC16IS740ExternalInterrupt( UINT32 param );
extern void UartSC16IS740RTS(BOOL state);
extern BOOL UartSC16IS740GetCTS( void );
extern void SC16IS740UartConsoleInit(void);
extern void CreateSC16IS740UartClass(const SC16IS740Cfg *pHwCfg);
extern INT32 SC1740DeviceProcessInput(UINT32 inst, UINT32 cmd, UINT8 * pData, UINT32 * pDataBytes);
extern void UartSC16IS740ConsoleTx(UINT8 *data, UINT32 count, UINT32 inst);
extern void UartSC16IS740HwUartTx(UINT32 uartNum, UINT8 data);
extern void UartSC16IS740HwConfig(UINT32 inst, UINT32 bps, UINT32 stopBits, UINT32 parity, UINT32 wlenght, UINT32 hwcontrol);
extern void UartSC16IS740CreateMsgQueue(void);
extern void UartSC16IS740DestroyMsgQueue(void);
extern void UartSC16IS740DisableHardware(void);
extern void UartSC16IS740ServiceUart(void);
// External UART Errors
typedef enum
{
  ERR_EXT_UART_TASK_NO_EVENT,
  ERR_EXT_UART_EVENT_QUEUE_FULL
    
}ERR_EXT_UART;

extern const SC16IS740Cfg SetupExtUartCfg;
extern const SC16IS740Cfg ExtUartCfg;

#define SC16IS740_SERVICE_QUEUE_PERIOD 10 //milliseconds
#endif  // _UART_SC16IS740_H_
